package org.msh.tb.export_rest.core;

import org.jboss.seam.ScopeType;
import org.jboss.seam.annotations.AutoCreate;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Scope;
import org.msh.tb.application.tasks.AsyncTask;
import org.msh.tb.application.tasks.TaskManager;
import org.msh.tb.application.tasks.TaskStatus;
import org.msh.tb.entities.TaskLog;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Support for asynchronous execution of the {@link ExcelExportAsyncService}
 * <p>
 * Created by rmemoria on 14/7/17.
 */
@Name("excelExportAsyncService")
@Scope(ScopeType.APPLICATION)
@AutoCreate
public class ExcelExportAsyncService {

    @In
    TaskManager taskManager;

    @In
    ExcelExportService excelExportService;

    /**
     * Start the exporting of data to an Excel file asynchronously. The method returns a
     * task identifier that is used by the caller to check if the exporting has finished
     *
     * @param file         the file that will be generated by the exporting service
     * @param masterReader
     * @param childReaders
     * @return task ID, used to check the status of the exporting
     * @throws IOException
     */
    public String startExport(File file, BlockReader masterReader,
                              List<BlockReader> childReaders) throws IOException {
        Map<String, Object> params = new HashMap<String, Object>();

        params.put(ExcelExportTask.PARAM_EXCEL_FILE, file);
        params.put(ExcelExportTask.PARAM_MASTER_READER, masterReader);
        params.put(ExcelExportTask.PARAM_CHILD_READERS, childReaders);
        params.put(ExcelExportTask.PARAM_EXCEL_EXPORT_SERVICE, excelExportService);
        params.put("locale", "en_NG");

        String taskId = taskManager.runTask(ExcelExportTask.class, params);

        return taskId;
    }

    public ExcelExportStatusResponse getStatus(String taskId) {
        AsyncTask task = taskManager.findTaskById(taskId);

        ExcelExportStatusResponse resp = new ExcelExportStatusResponse();

        if (task != null) {
            double prog = task.getProgress();
            resp.setProgress(prog);
            resp.setStatus(TaskStatus.RUNNING);
            return resp;
        }

        TaskLog taskLog = taskManager.findTaskLog(taskId);
        if (taskLog == null) {
            resp.setStatus(TaskStatus.UNKNOWN);
            return resp;
        }

        resp.setStatus(taskLog.getStatus());

        return resp;
    }

    /**
     * Get the Excel file reference from the finished exporting task
     *
     * @param taskId
     * @return
     */
    public File getDownloadFile(String taskId) {
        TaskLog taskLog = taskManager.findTaskLog(taskId);
        if (taskLog == null || taskLog.getStatus() != TaskStatus.FINISHED) {
            return null;
        }

        File excelFile = new File(taskLog.getData());

        return excelFile.exists() ? excelFile : null;
    }

    /**
     * Cancel an on going exporting
     *
     * @param taskId
     */
    public boolean cancelExport(String taskId) {
        return taskManager.cancelTask(taskId);
    }
}